/* Copyright (C) 2016-2018 RealVNC Ltd.  All Rights Reserved.
 */

/* This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "GStreamerDecoder.h"

#include <cstring>

GStreamerDecoder::GStreamerDecoder(
        BaseDecoderSupport& support,
        VNCCommonDecoderApplicationContext context)

    :   mSupport(support),
        mContext(context),
        mLog(support.log().tag("GStreamerDecoder")),
        mOverlayVisible(false),
        mVolume(50)
{
    memset(&mRectangle, 0, sizeof(mRectangle));
}

std::string GStreamerDecoder::getSupportInfoDecoderName()
{
    return "GStreamer Decoder";
}

std::string GStreamerDecoder::getSupportInfoDecoderVersion()
{
    return "v1.0";
}

bool GStreamerDecoder::getCreationParamDisableBuffering()
{
    return false;
}

VNCCommonDecoderError GStreamerDecoder::setPropertyGeneric(
        const VNCCommonDecoderPropertyKey propertyKey,
        const VNCCommonDecoderDataType propertyType,
        void *propertyValue)
{

    vnccommon::Mutex::Locker lock(mMutex);

    if (propertyKey == VNCCommonDecoderPropertyKeyVENDOR + 1 &&
        propertyType == VNCCommonDecoderDataTypeU32)
    {
        setVolume(*static_cast<vnc_uint32_t*>(propertyValue));
    }
    return VNCCommonDecoderErrorNone;
}

VNCCommonDecoderError GStreamerDecoder::setPropertyBool(
        const VNCCommonDecoderPropertyKey propertyKey,
        const bool propertyValue)
{
    vnccommon::Mutex::Locker lock(mMutex);

    if (propertyKey == VNCCommonDecoderPropertyKeyOverlayVisibility)
    {
        mLog.info("Overlay %s", (propertyValue ? "VISIBLE" : "HIDDEN"));
        mOverlayVisible = propertyValue;

        for (std::map<vnc_uint64_t, GStreamerStream*>::iterator it = mStreams.begin();
             it != mStreams.end();
             ++it)
        {
            it->second->setOverlayVisibility(propertyValue);
        }
    }
    else if (propertyKey == VNCCommonDecoderPropertyKeyVENDOR + 2)
    {
        setMuted(propertyValue);
    }

    return VNCCommonDecoderErrorNone;
}

VNCCommonDecoderError GStreamerDecoder::setPropertyRectangle(
        const VNCCommonDecoderPropertyKey propertyKey,
        const VNCRectangle& propertyValue)
{
    vnccommon::Mutex::Locker lock(mMutex);

    if (propertyKey == VNCCommonDecoderPropertyKeyOverlayRectangle)
    {
        mLog.info("Overlay rectangle set (%d, %d) - (%d, %d)",
            propertyValue.topLeft.x,
            propertyValue.topLeft.y,
            propertyValue.bottomRight.x,
            propertyValue.bottomRight.y);
        for (std::map<vnc_uint64_t, GStreamerStream*>::iterator it = mStreams.begin();
             it != mStreams.end();
             ++it)
        {
            it->second->setOverlayRectangle(propertyValue);
        }
        mRectangle = propertyValue;
    }

    return VNCCommonDecoderErrorNone;
}

VNCCommonDecoderError GStreamerDecoder::setPropertyString(
        const VNCCommonDecoderPropertyKey propertyKey,
        const std::string& propertyValue)
{
    (void) propertyKey;
    (void) propertyValue;

    return VNCCommonDecoderErrorNone;
}

void GStreamerDecoder::getEventHandles(
        std::vector<VNCCommonDecoderEventHandleParams>& out_EventHandles)
{
    const VNCCommonDecoderEventHandleParams handle =
    {
            mQueue.getEventHandle(),
            true,
            false
    };

    out_EventHandles.push_back(handle);
}

void GStreamerDecoder::notifyEventHandleActivity(
        const std::vector<VNCCommonDecoderEventHandle>&)
{
    vnccommon::OptionalAutoPtr<ControlMessage> msg;

    while((msg = mQueue.tryDequeue()).hasValue())
    {
        msg.value().run(mSupport);
    }
}

VNCCommonDecoderError GStreamerDecoder::querySupportAudioLPCM(
        const VNCCommonDecoderAudioFormatDetailsLPCM& formatDetails,
        bool& out_isEncodingSupported,
        std::vector<VNCCommonDecoderAudioFormatDetailsLPCM>&)
{
    (void) formatDetails;

    // The GStreamer decoder supports everything
    out_isEncodingSupported = true;

    return VNCCommonDecoderErrorNone;
}

VNCCommonDecoderError GStreamerDecoder::querySupportVideoH264(
        const vnc_size_t h264AttributesSize,
        const VNCCommonDecoderH264Attributes& h264Attributes,
        const std::vector<VNCCommonDecoderVideoMode>& videoModes,
        std::vector<bool>& out_isEncodingSupported,
        std::vector<VNCCommonDecoderVideoMode>&)
{
    (void) videoModes;

    VNCCommonDecoderH264Attributes h264AttributesSafe;
    ::memset(&h264AttributesSafe, 0, sizeof(h264AttributesSafe));

    VNCCOMMON_SAFE_MEMCPY_STRUCT(
            VNCCommonDecoderH264Attributes,
            h264AttributesSize,
            vnc_int32_t,
            levelMinor,
            &h264AttributesSafe,
            &h264Attributes);

    // We support them ALL!
    for(size_t i = 0; i < out_isEncodingSupported.size(); ++i)
    {
        out_isEncodingSupported[i] = true;
    }

    return VNCCommonDecoderErrorNone;
}

void GStreamerDecoder::streamCreated(
        const StreamID streamId,
        const VNCCommonDecoderMediaType mediaType,
        const VNCCommonDecoderStreamSpecificType specificType)
{
    (void) specificType;

    mStreams.insert(
            std::make_pair(
                    streamId.toInteger(),
                    new GStreamerStream(streamId, mLog, mediaType, mContext)));

    mQueue.enqueue(std::auto_ptr<ControlMessage>(
            new ControlMessageStreamCreateCompleted(streamId)));
}

void GStreamerDecoder::streamStarted(
        const StreamID streamId)
{
    GStreamerStream* stream = mStreams[streamId.toInteger()];
    stream->setupGStreamer();
    stream->setVolume(mVolume);
    stream->start();

    mQueue.enqueue(std::auto_ptr<ControlMessage>(
            new ControlMessageStreamStartCompleted(streamId)));
}

void GStreamerDecoder::streamStopped(
        const StreamID streamId,
        const bool /* stopImmediately */)
{
    mStreams[streamId.toInteger()]->stop();

    mQueue.enqueue(std::auto_ptr<ControlMessage>(
            new ControlMessageStreamStopCompleted(streamId)));
}

void GStreamerDecoder::streamDestroyed(
        const StreamID streamId)
{
    vnccommon::Mutex::Locker lock(mMutex);

    delete mStreams[streamId.toInteger()];
    mStreams.erase(streamId.toInteger());

    mQueue.enqueue(std::auto_ptr<ControlMessage>(
            new ControlMessageStreamDestroyCompleted(streamId)));
}

void GStreamerDecoder::streamHintAudioContent(
        const StreamID streamId,
        const std::vector<VNCCommonDecoderAudioStreamContentType>& audioStreamContentTypes)
{
    (void) streamId;
    (void) audioStreamContentTypes;
}

void GStreamerDecoder::streamHintAudioEncodingLPCM(
        const StreamID streamId,
        const VNCCommonDecoderAudioFormatDetailsLPCM& formatDetails)
{
    mStreams[streamId.toInteger()]->setAudioFormat(formatDetails);
}

void GStreamerDecoder::streamHintVideoEncodingH264(
        const StreamID streamId,
        const vnc_size_t h264AttributesSize,
        const VNCCommonDecoderH264Attributes& h264Attributes,
        const vnccommon::Optional<VNCCommonDecoderVideoMode>& videoMode)
{
    (void) h264AttributesSize;
    (void) h264Attributes;

    if (videoMode.hasValue())
    {
        mStreams[streamId.toInteger()]->setVideoMode(videoMode.value());
    }
}

void GStreamerDecoder::streamPayload(
        const StreamID streamId,
        const vnc_size_t payloadStructSize,
        VNCCommonDecoderStreamPayload *const payload)
{
    (void) payloadStructSize;

    mStreams[streamId.toInteger()]->streamPayload(
            payload->payloadData, payload->payloadDataLength);

    mQueue.enqueue(std::auto_ptr<ControlMessage>(
            new ControlMessageStreamRecyclePayload(
                    streamId,
                    payload)));
}

void GStreamerDecoder::streamMuteEnable(
        const StreamID streamId)
{
    mStreams[streamId.toInteger()]->setMuted(true);
}

void GStreamerDecoder::streamMuteDisable(
        const StreamID streamId)
{
    mStreams[streamId.toInteger()]->setMuted(false);
}

void GStreamerDecoder::streamDuckEnable(
        const StreamID streamId,
        const vnccommon::Optional<vnc_uint64_t>& suggestedRampMs,
        const vnccommon::Optional<vnc_uint32_t>& suggestedAttenuationDb)
{
    (void) suggestedRampMs;
    (void) suggestedAttenuationDb;

    mStreams[streamId.toInteger()]->setVolume(mVolume * 0.5);
}

void GStreamerDecoder::streamDuckDisable(
        const StreamID streamId)
{
    mStreams[streamId.toInteger()]->setVolume(mVolume);
}

void GStreamerDecoder::streamNotifyFramerateLimitRequestSuccess(
        const StreamID streamId,
        const vnccommon::Optional<vnc_uint32_t>& newFrameRate)
{
    (void) streamId;
    (void) newFrameRate;
}

void GStreamerDecoder::destroy()
{
}

GStreamerDecoder::~GStreamerDecoder()
{
}

void GStreamerDecoder::setVolume(vnc_uint32_t volume)
{
    mVolume = volume;
    mLog.info("Volume %d", mVolume);
    for (std::map<vnc_uint64_t, GStreamerStream*>::iterator it = mStreams.begin(); it != mStreams.end(); ++it)
    {
        it->second->setVolume(mVolume);
    }
}

void GStreamerDecoder::setMuted(bool muted)
{
    mMuted = muted;
    mLog.info("Muted %s", mMuted ? "TRUE" : "FALSE");
    for (std::map<vnc_uint64_t, GStreamerStream*>::iterator it = mStreams.begin(); it != mStreams.end(); ++it)
    {
        it->second->setMuted(mMuted);
    }
}

std::auto_ptr<BaseDecoderImpl> instantiateDecoder(
        VNCCommonDecoderApplicationContext context,
        const VNCCommonDecoderSDKAttributes&,
        BaseDecoderSupport& support)
{
    return std::auto_ptr<BaseDecoderImpl>(new GStreamerDecoder(
            support, context));
}

